{%- if cookiecutter.enable_ai_agent %}
"""AI Agent WebSocket routes with streaming support."""

import logging
from typing import Any
{%- if cookiecutter.enable_conversation_persistence and cookiecutter.use_database %}
from datetime import datetime, UTC
{%- if cookiecutter.use_postgresql %}
from uuid import UUID
{%- endif %}
{%- endif %}

from fastapi import APIRouter, WebSocket, WebSocketDisconnect{%- if cookiecutter.websocket_auth_jwt %}, Depends{%- endif %}{%- if cookiecutter.websocket_auth_api_key %}, Query{%- endif %}

from pydantic_ai import (
    Agent,
    FinalResultEvent,
    FunctionToolCallEvent,
    FunctionToolResultEvent,
    PartDeltaEvent,
    PartStartEvent,
    TextPartDelta,
    ToolCallPartDelta,
)
from pydantic_ai.messages import (
    ModelRequest,
    ModelResponse,
    SystemPromptPart,
    TextPart,
    UserPromptPart,
)

from app.agents.assistant import Deps, get_agent
{%- if cookiecutter.websocket_auth_jwt %}
from app.api.deps import get_current_user_ws
from app.db.models.user import User
{%- endif %}
{%- if cookiecutter.websocket_auth_api_key %}
from app.core.config import settings
{%- endif %}
{%- if cookiecutter.enable_conversation_persistence and (cookiecutter.use_postgresql or cookiecutter.use_sqlite) %}
from app.db.session import get_db_session
from app.api.deps import ConversationSvc, get_conversation_service
from app.schemas.conversation import ConversationCreate, MessageCreate, ToolCallCreate, ToolCallComplete
{%- elif cookiecutter.enable_conversation_persistence and cookiecutter.use_mongodb %}
from app.api.deps import ConversationSvc, get_conversation_service
from app.schemas.conversation import ConversationCreate, MessageCreate, ToolCallCreate, ToolCallComplete
{%- endif %}

logger = logging.getLogger(__name__)

router = APIRouter()


class AgentConnectionManager:
    """WebSocket connection manager for AI agent."""

    def __init__(self) -> None:
        self.active_connections: list[WebSocket] = []

    async def connect(self, websocket: WebSocket) -> None:
        """Accept and store a new WebSocket connection."""
        await websocket.accept()
        self.active_connections.append(websocket)
        logger.info(f"Agent WebSocket connected. Total connections: {len(self.active_connections)}")

    def disconnect(self, websocket: WebSocket) -> None:
        """Remove a WebSocket connection."""
        if websocket in self.active_connections:
            self.active_connections.remove(websocket)
        logger.info(f"Agent WebSocket disconnected. Total connections: {len(self.active_connections)}")

    async def send_event(self, websocket: WebSocket, event_type: str, data: Any) -> bool:
        """Send a JSON event to a specific WebSocket client.

        Returns True if sent successfully, False if connection is closed.
        """
        try:
            await websocket.send_json({"type": event_type, "data": data})
            return True
        except (WebSocketDisconnect, RuntimeError):
            # Connection already closed
            return False


manager = AgentConnectionManager()


def build_message_history(history: list[dict[str, str]]) -> list[ModelRequest | ModelResponse]:
    """Convert conversation history to PydanticAI message format."""
    model_history: list[ModelRequest | ModelResponse] = []

    for msg in history:
        if msg["role"] == "user":
            model_history.append(ModelRequest(parts=[UserPromptPart(content=msg["content"])]))
        elif msg["role"] == "assistant":
            model_history.append(ModelResponse(parts=[TextPart(content=msg["content"])]))
        elif msg["role"] == "system":
            model_history.append(ModelRequest(parts=[SystemPromptPart(content=msg["content"])]))

    return model_history

{%- if cookiecutter.websocket_auth_api_key %}


async def verify_api_key(api_key: str) -> bool:
    """Verify the API key for WebSocket authentication."""
    return api_key == settings.API_KEY
{%- endif %}


@router.websocket("/ws/agent")
async def agent_websocket(
    websocket: WebSocket,
{%- if cookiecutter.websocket_auth_jwt %}
    user: User = Depends(get_current_user_ws),
{%- elif cookiecutter.websocket_auth_api_key %}
    api_key: str = Query(..., alias="api_key"),
{%- endif %}
) -> None:
    """WebSocket endpoint for AI agent with full event streaming.

    Uses PydanticAI iter() to stream all agent events including:
    - user_prompt: When user input is received
    - model_request_start: When model request begins
    - text_delta: Streaming text from the model
    - tool_call_delta: Streaming tool call arguments
    - tool_call: When a tool is called (with full args)
    - tool_result: When a tool returns a result
    - final_result: When the final result is ready
    - complete: When processing is complete
    - error: When an error occurs

    Expected input message format:
    {
        "message": "user message here",
        "history": [{"role": "user|assistant|system", "content": "..."}]{% if cookiecutter.enable_conversation_persistence and cookiecutter.use_database %},
        "conversation_id": "optional-uuid-to-continue-existing-conversation"{% endif %}
    }
{%- if cookiecutter.websocket_auth_jwt %}

    Authentication: Requires a valid JWT token passed as a query parameter or header.
{%- elif cookiecutter.websocket_auth_api_key %}

    Authentication: Requires a valid API key passed as 'api_key' query parameter.
    Example: ws://localhost:{{ cookiecutter.backend_port }}/api/v1/ws/agent?api_key=your-api-key
{%- endif %}
{%- if cookiecutter.enable_conversation_persistence and cookiecutter.use_database %}

    Persistence: Set 'conversation_id' to continue an existing conversation.
    If not provided, a new conversation is created. The conversation_id is
    returned in the 'conversation_started' event.
{%- endif %}
    """
{%- if cookiecutter.websocket_auth_api_key %}
    # Verify API key before accepting connection
    if not await verify_api_key(api_key):
        await websocket.close(code=4001, reason="Invalid API key")
        return
{%- endif %}

    await manager.connect(websocket)

    # Conversation state per connection
    conversation_history: list[dict[str, str]] = []
    deps = Deps()
{%- if cookiecutter.enable_conversation_persistence and cookiecutter.use_database %}
    current_conversation_id: str | None = None
{%- endif %}

    try:
        while True:
            # Receive user message
            data = await websocket.receive_json()
            user_message = data.get("message", "")
            # Optionally accept history from client (or use server-side tracking)
            if "history" in data:
                conversation_history = data["history"]

            if not user_message:
                await manager.send_event(websocket, "error", {"message": "Empty message"})
                continue

{%- if cookiecutter.enable_conversation_persistence and (cookiecutter.use_postgresql or cookiecutter.use_sqlite) %}

            # Handle conversation persistence
            db_gen = get_db_session()
            db = await anext(db_gen) if hasattr(db_gen, "__anext__") else next(db_gen)
            try:
                conv_service = get_conversation_service(db)

                # Get or create conversation
                requested_conv_id = data.get("conversation_id")
                if requested_conv_id:
{%- if cookiecutter.use_postgresql %}
                    current_conversation_id = requested_conv_id
                    # Verify conversation exists
                    await conv_service.get_conversation(UUID(requested_conv_id))
{%- else %}
                    current_conversation_id = requested_conv_id
                    conv_service.get_conversation(requested_conv_id)
{%- endif %}
                elif not current_conversation_id:
                    # Create new conversation
                    conv_data = ConversationCreate(
{%- if cookiecutter.use_jwt %}
                        user_id={% if cookiecutter.use_postgresql %}user.id{% else %}str(user.id){% endif %},
{%- endif %}
                        title=user_message[:50] if len(user_message) > 50 else user_message,
                    )
{%- if cookiecutter.use_postgresql %}
                    conversation = await conv_service.create_conversation(conv_data)
{%- else %}
                    conversation = conv_service.create_conversation(conv_data)
{%- endif %}
                    current_conversation_id = str(conversation.id)
                    await manager.send_event(
                        websocket,
                        "conversation_started",
                        {"conversation_id": current_conversation_id},
                    )

                # Save user message
{%- if cookiecutter.use_postgresql %}
                user_msg = await conv_service.add_message(
                    UUID(current_conversation_id),
                    MessageCreate(role="user", content=user_message),
                )
{%- else %}
                user_msg = conv_service.add_message(
                    current_conversation_id,
                    MessageCreate(role="user", content=user_message),
                )
{%- endif %}
                await db.commit()
            except Exception as e:
                logger.warning(f"Failed to persist conversation: {e}")
                # Continue without persistence
{%- elif cookiecutter.enable_conversation_persistence and cookiecutter.use_mongodb %}

            # Handle conversation persistence (MongoDB)
            conv_service = get_conversation_service()

            requested_conv_id = data.get("conversation_id")
            if requested_conv_id:
                current_conversation_id = requested_conv_id
                await conv_service.get_conversation(requested_conv_id)
            elif not current_conversation_id:
                conv_data = ConversationCreate(
{%- if cookiecutter.use_jwt %}
                    user_id=str(user.id),
{%- endif %}
                    title=user_message[:50] if len(user_message) > 50 else user_message,
                )
                conversation = await conv_service.create_conversation(conv_data)
                current_conversation_id = str(conversation.id)
                await manager.send_event(
                    websocket,
                    "conversation_started",
                    {"conversation_id": current_conversation_id},
                )

            # Save user message
            user_msg = await conv_service.add_message(
                current_conversation_id,
                MessageCreate(role="user", content=user_message),
            )
{%- endif %}

            await manager.send_event(websocket, "user_prompt", {"content": user_message})

            try:
                assistant = get_agent()
                model_history = build_message_history(conversation_history)

                # Use iter() on the underlying PydanticAI agent to stream all events
                async with assistant.agent.iter(
                    user_message,
                    deps=deps,
                    message_history=model_history,
                ) as agent_run:
                    async for node in agent_run:
                        if Agent.is_user_prompt_node(node):
                            await manager.send_event(
                                websocket,
                                "user_prompt_processed",
                                {"prompt": node.user_prompt},
                            )

                        elif Agent.is_model_request_node(node):
                            await manager.send_event(websocket, "model_request_start", {})

                            async with node.stream(agent_run.ctx) as request_stream:
                                async for event in request_stream:
                                    if isinstance(event, PartStartEvent):
                                        await manager.send_event(
                                            websocket,
                                            "part_start",
                                            {
                                                "index": event.index,
                                                "part_type": type(event.part).__name__,
                                            },
                                        )
                                        # Send initial content from TextPart if present
                                        if isinstance(event.part, TextPart) and event.part.content:
                                            await manager.send_event(
                                                websocket,
                                                "text_delta",
                                                {
                                                    "index": event.index,
                                                    "content": event.part.content,
                                                },
                                            )

                                    elif isinstance(event, PartDeltaEvent):
                                        if isinstance(event.delta, TextPartDelta):
                                            await manager.send_event(
                                                websocket,
                                                "text_delta",
                                                {
                                                    "index": event.index,
                                                    "content": event.delta.content_delta,
                                                },
                                            )
                                        elif isinstance(event.delta, ToolCallPartDelta):
                                            await manager.send_event(
                                                websocket,
                                                "tool_call_delta",
                                                {
                                                    "index": event.index,
                                                    "args_delta": event.delta.args_delta,
                                                },
                                            )

                                    elif isinstance(event, FinalResultEvent):
                                        await manager.send_event(
                                            websocket,
                                            "final_result_start",
                                            {"tool_name": event.tool_name},
                                        )

                        elif Agent.is_call_tools_node(node):
                            await manager.send_event(websocket, "call_tools_start", {})

                            async with node.stream(agent_run.ctx) as handle_stream:
                                async for event in handle_stream:
                                    if isinstance(event, FunctionToolCallEvent):
                                        await manager.send_event(
                                            websocket,
                                            "tool_call",
                                            {
                                                "tool_name": event.part.tool_name,
                                                "args": event.part.args,
                                                "tool_call_id": event.part.tool_call_id,
                                            },
                                        )

                                    elif isinstance(event, FunctionToolResultEvent):
                                        await manager.send_event(
                                            websocket,
                                            "tool_result",
                                            {
                                                "tool_call_id": event.tool_call_id,
                                                "content": str(event.result.content),
                                            },
                                        )

                        elif Agent.is_end_node(node) and agent_run.result is not None:
                            await manager.send_event(
                                websocket,
                                "final_result",
                                {"output": agent_run.result.output},
                            )

                # Update conversation history
                conversation_history.append({"role": "user", "content": user_message})
                if agent_run.result:
                    conversation_history.append(
                        {"role": "assistant", "content": agent_run.result.output}
                    )

{%- if cookiecutter.enable_conversation_persistence and (cookiecutter.use_postgresql or cookiecutter.use_sqlite) %}

                # Save assistant response to database
                if current_conversation_id and agent_run.result:
                    try:
{%- if cookiecutter.use_postgresql %}
                        await conv_service.add_message(
                            UUID(current_conversation_id),
                            MessageCreate(
                                role="assistant",
                                content=agent_run.result.output,
                                model_name=assistant.model_name if hasattr(assistant, "model_name") else None,
                            ),
                        )
                        await db.commit()
{%- else %}
                        conv_service.add_message(
                            current_conversation_id,
                            MessageCreate(
                                role="assistant",
                                content=agent_run.result.output,
                                model_name=assistant.model_name if hasattr(assistant, "model_name") else None,
                            ),
                        )
                        db.commit()
{%- endif %}
                    except Exception as e:
                        logger.warning(f"Failed to persist assistant response: {e}")
{%- elif cookiecutter.enable_conversation_persistence and cookiecutter.use_mongodb %}

                # Save assistant response to database
                if current_conversation_id and agent_run.result:
                    try:
                        await conv_service.add_message(
                            current_conversation_id,
                            MessageCreate(
                                role="assistant",
                                content=agent_run.result.output,
                                model_name=assistant.model_name if hasattr(assistant, "model_name") else None,
                            ),
                        )
                    except Exception as e:
                        logger.warning(f"Failed to persist assistant response: {e}")
{%- endif %}

                await manager.send_event(websocket, "complete", {
{%- if cookiecutter.enable_conversation_persistence and cookiecutter.use_database %}
                    "conversation_id": current_conversation_id,
{%- endif %}
                })

            except WebSocketDisconnect:
                # Client disconnected during processing - this is normal
                logger.info("Client disconnected during agent processing")
                break
            except Exception as e:
                logger.exception(f"Error processing agent request: {e}")
                # Try to send error, but don't fail if connection is closed
                await manager.send_event(websocket, "error", {"message": str(e)})

    except WebSocketDisconnect:
        pass  # Normal disconnect
    finally:
        manager.disconnect(websocket)
{%- else %}
"""AI Agent routes - not configured."""
{%- endif %}
